/*
#
# Worker for writing GOP information to files
#
# $Id: GOPutils.cpp,v 1.2 2003/06/07 10:03:16 nemies Exp $
#
# Copyright (C) 2003 Kees Cook
# kees@outflux.net, http://outflux.net/
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# http://www.gnu.org/copyleft/gpl.html
#
*/

#include "GOPchop.h"
#include "GOPutils.h"

#include "MPEG2Parser.h"
#include "Picture.h"

/*
 * This exports a text file that can be used to re-load an MPEG2-PS
 * much more quickly.  (And this file can be used to check the results
 * of GOPchop's parser.)
 *
 * Format is:
 *   TAG COUNT OFFSET[:LENGTH]
 *
 * GOP %u %llu:%lu
 * pack %u %llu
 * ...
 * pic %u %llu:%lu
 * ves %u %llu:%lu
 * ...
 *
 * If you have more GOPs than can be indexed by a 32bit variable,
 * then you can change the index variable yourself.  ;)
 *
 */
int write_gop_cache(char * filename, MPEG2Parser * mpeg2parser)
{
    int result=-1;
    List *GOPs;
    List *packets;
    List *pictures;
    List *ves;
    Vector *vector;
    uint32_t gopnum, gopmax;
    FILE * fp;

    if (!(fp=fopen(filename,"w"))) {
        perror(filename);
        return -1;
    }

    if (!(GOPs = mpeg2parser->getGOPs())) {
        printf("%s",_("NULL GOP list?!\n"));
        goto need_close;
    }
    if (!(packets = mpeg2parser->getPackets())) {
        printf("%s", _("NULL packet list?!\n"));
        goto need_close;
    }
    if (!(pictures = mpeg2parser->getPictures())) {
        printf("%s", _("NULL picture list?!\n"));
        goto need_close;
    }
    if (!(ves = mpeg2parser->getVideo())) {
        printf("%s", _("NULL VES list?!\n"));
        goto need_close;
    }

    gopmax=GOPs->getNum();
    for (gopnum=0;gopnum<gopmax;gopnum++) {
        GroupOfPictures *gop;
        Bounds *bounds;
        uint32_t num, max;
        Vector *header;

        if (!(gop=(GroupOfPictures*)GOPs->vector(gopnum))) {
            printf("%s", _("Missing GOP %u!?\n"),gopnum);
            goto need_close;
        }

        if (!(header=gop->getHeader())) {
            printf("%s",_("Missing GOP %u header!?\n"),gopnum);
            goto need_close;
        }

        fprintf(fp,"GOP %u %" OFF_T_FORMAT "\n",
                   gopnum, header->getStart());

        /* report packets */
        if (!(bounds=gop->getPacketBounds())) {
            printf("%s", _("NULL packet list for GOP %d?!\n"), gopnum);
            goto need_close;
        }
        max=bounds->getMax();
        for (num=bounds->getFirst();num<max;num++) {
            Pack *pack;

            if (!(pack=(Pack*)packets->vector(num))) {
                printf("%s",_("Missing Pack %u!?\n"),num);
                goto need_close;
            }

            fprintf(fp,"pack %u %" OFF_T_FORMAT "\n",
                       num, pack->getStart());
            
        }

        /* report pictures */
        if (!(bounds=gop->getPictureBounds())) {
            printf("%s", _("NULL picture list for GOP %d?!\n"), gopnum);
            goto need_close;
        }
        max=bounds->getMax();
        for (num=bounds->getFirst();num<max;num++) {
            Picture *picture;
            Bounds * vesbounds;
            uint32_t vesnum, vesmax;

            if (!(picture=(Picture*)pictures->vector(num))) {
                printf("%s",_("Missing Picture %u!?\n"),num);
                goto need_close;
            }

            fprintf(fp,"pic %u %" OFF_T_FORMAT "\n",
                       num,picture->getStart());

            /* report video */
            if (!(vesbounds=picture->getVideoBounds())) {
                printf("%s", _("NULL video ES list for GOP %d?!\n"), gopnum);
                goto need_close;
            }
            vesmax=vesbounds->getMax();
            for (vesnum=vesbounds->getFirst();vesnum<vesmax;vesnum++) {
                ElementStream *es;

                if (!(es=(ElementStream*)ves->vector(vesnum))) {
                    printf("%s",_("Missing VES packet %u!?\n"),vesnum);
                    goto need_close;
                }

                fprintf(fp,"ves %u %" OFF_T_FORMAT ":%lu\n",
                           vesnum,es->getStart(),es->getLen());
            }
        }
    }

    result=0;

need_close:
    if (fclose(fp)) {
        perror(filename);
        result=-1;
    }
    return result;
}

void write_clip_list(char * filename, struct clip_list_t * clips)
{
}

/* vi:set ai ts=4 sw=4 expandtab: */
