/*
#
# Tool for displaying SPU graphics from DVD navigation streams
#
# $Id: gtkspu.c,v 1.5 2003/04/27 03:08:37 nemies Exp $
#
# Copyright (C) 2002-2003 Kees Cook
# kees@outflux.net, http://outflux.net/
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# http://www.gnu.org/copyleft/gpl.html
#
*/

#include "GOPchop.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

typedef struct
{
    gint red;
    gint green;
    gint blue;
}
quick_color;

quick_color clut[0xF];

#define COLOR_BACK      0
#define COLOR_PATTERN   1
#define COLOR_EMP1      2
#define COLOR_EMP2      3
#define COLOR_NOTHING   4

gint width = 0;
gint height = 0;
guchar *rgbbuf = NULL;
guchar *backbuf = NULL;
int *spu_pixelbuf = NULL;

int clut_index[4];
int alpha[5] = { 0x0, 0x0, 0x0, 0x0, 0x0 };     // all transparent
quick_color cmap[4];

gboolean on_darea_expose(GtkWidget * widget,
                         GdkEventExpose * event, gpointer user_data);

int yank_nibble(guchar * rledata, int *addr, int *halfbyte)
{
    int val = 0;

    if (!*halfbyte)
    {
        // take the top half
        val = (rledata[*addr] & 0xF0) >> 4;

        *halfbyte = 1;
    }
    else
    {
        // take the bottom half
        val = rledata[*addr] & 0x0F;

        *halfbyte = 0;
        *addr += 1;
    }
    return val;
}

#define GRAB_NIBBLE             yank_nibble(rledata,&addr,&halfbyte)


void draw_spu(int *drawbuf,
              int row_start, int row_step,
              int colstart, guchar * rledata, int addr)
{
    int row;
    int col = colstart;

    int bits;
    int halfbyte = 0;
    int number;
    int cindex;

    for (row = row_start; row < height;)
    {
        bits = GRAB_NIBBLE;
        if ((bits & 0xC) != 0x0)
        {
            // have 4-bit code
            number = (bits & 0xC) >> 2;
            cindex = (bits & 0x3);
        }
        else
        {
            bits <<= 4;
            bits |= GRAB_NIBBLE;

            if ((bits & 0xF0) != 0)
            {
                // have 8-bit code
                number = (bits & 0x3C) >> 2;
                cindex = (bits & 0x3);
            }
            else
            {
                bits <<= 4;
                bits |= GRAB_NIBBLE;

                if ((bits & 0xFC0) != 0)
                {
                    // have 12-bit code
                    number = (bits & 0xFC) >> 2;
                    cindex = (bits & 0x3);
                }
                else
                {
                    // have 16-bit code
                    bits <<= 4;
                    bits |= GRAB_NIBBLE;

                    number = (bits & 0x3FC) >> 2;
                    cindex = (bits & 0x3);

                    if (number == 0)
                        number = width; // entire row
                }
            }
        }


        // write "number" many "cindex"s until end of row
        //printf("row: %d col: %d cindex: %d\n", row,col,cindex);
        for (; number > 0 && col < width; number--)
        {
            drawbuf[row * width + col++] = cindex;
        }

        if (col >= width)
        {
            row += row_step;
            col = colstart;

            // fill to end of byte on row end
            if (halfbyte)
            {
                halfbyte = 0;
                addr++;
            }
        }
    }

}

#define SPU_CMD_END                     0xFF    // ends one SP_DCSQ
#define SPU_FORCED_START_DISPLAY        0x00    // no arguments
#define SPU_START_DISPLAY               0x01    // no arguments
#define SPU_STOP_DISPLAY                0x02    // no arguments

// provides four indices into the CLUT for the current PGC to associate with
// the four pixel values. One nibble per pixel value for a total of 2 bytes: 
// emphasis2 emphasis1 pattern background
#define SPU_SET_COLOR                   0x03

// directly provides the four contrast (alpha blend) values to associate with
// the four pixel values. One nibble per pixel value for a total of 2 bytes.
// 0x0 = transparent, 0xF = opaque
// e2 e1   p b 
#define SPU_SET_CONTRAST                0x04

// defines the display area, each pair (X and Y) of values is 3 bytes wide,
// for a total of 6 bytes, and has the form:
// sx sx   sx ex   ex ex   sy sy   sy ey   ey ey
// sx = starting X coordinate
// ex = ending X coordinate
// sy = starting Y coordinate
// ey = ending Y coordinate 
#define SPU_DEFINE_AREA                 0x05

// defines the pixel data addresses. First a 2-byte offset to the top field
// data, followed by a 2-byte offset to the bottom field data, for a total of
// 4 bytes.
#define SPU_DEFINE_PIXEL_ADDRESS        0x06

// allows for changing the COLor and CONtrast within one or more areas of the
// display. This command contains a series of parameters, arranged in a
// hierarchy.
// Following the command byte is a 2-byte value for the total size of the
// parameter area, including the size word.
//
// The parameter sequence begins with a LN_CTLI, which defines a vertically
// bounded area of the display. The LN_CTLI may include from one to fifteen
// PX_CTLI parameters, which define a starting horizontal position and new
// color and contrast value to apply from that column on towards the right to
// the next PX_CTLI or the right side of the display. 
//
// LN_CTLI, 4 bytes, special value of 0f ff ff ff signifies the end of the
// parameter area (this termination code MUST be present as the last parameter)
// 0 s   s s   n t   t t
// sss = csln, the starting (top-most) line number for this area
// n = number of PX_CTLI to follow
// ttt = ctln, the terminating (bottom-most) line number for this area
//
// PX_CTLI, 6 bytes, defines a starting column and new color and contrast values
// bytes 0 and 1 - starting column number
// bytes 2 and 3 - new color values, as per SET_COLOR
// bytes 4 and 5 - new contrast values, as per SET_CONTR 
#define SPU_CHANGE_COLOR_CONTRAST       0x07


#define GRAB_BYTE               (buf[location])
#define GRAB_WORD               ((buf[location]<<8)|buf[location+1])
#define GRAB_HIGH_NIBBLE        ((buf[location]&0xF0)>>4)
#define GRAB_LOW_NIBBLE         ((buf[location]&0x0F))


unsigned int odd_pixel_rows;
unsigned int even_pixel_rows;
gint sx, sy, ex, ey, i;

// returns -1 if done, or offset of next sequence
int spu_run(guchar * buf, int location)
{
    int my_location = location;
    unsigned int stall_time;
    unsigned int next_sequence;
    unsigned int size;

    guchar cmd;

    printf("0x%05X: ", location);
    stall_time = GRAB_WORD;
    location += 2;
    next_sequence = GRAB_WORD;
    printf("SEQ: Stall time: %u, Next sequence: 0x%X\n",
           stall_time, next_sequence);
    location += 2;

    for (cmd = buf[location];; cmd = buf[location])
    {
        printf("0x%05X: ", location);
        location++;
        switch (cmd)
        {
            case SPU_FORCED_START_DISPLAY:
                printf("Forced Start Display\n");
                break;
            case SPU_START_DISPLAY:
                printf("Start Display\n");
                break;
            case SPU_STOP_DISPLAY:
                printf("Stop Display\n");
                break;

            case SPU_SET_COLOR:
                clut_index[COLOR_EMP2] = GRAB_HIGH_NIBBLE;
                clut_index[COLOR_EMP1] = GRAB_LOW_NIBBLE;
                location++;
                clut_index[COLOR_PATTERN] = GRAB_HIGH_NIBBLE;
                clut_index[COLOR_BACK] = GRAB_LOW_NIBBLE;
                location++;
                printf
                    ("Set Color (pat: 0x%X back: 0x%X emp1: 0x%X emp2: 0x%X)\n",
                     clut_index[COLOR_PATTERN], clut_index[COLOR_BACK],
                     clut_index[COLOR_EMP1], clut_index[COLOR_EMP2]);

                cmap[0] = clut[clut_index[0]];
                cmap[1] = clut[clut_index[1]];
                cmap[2] = clut[clut_index[2]];
                cmap[3] = clut[clut_index[3]];

                break;

            case SPU_SET_CONTRAST:
                alpha[COLOR_EMP2] = GRAB_HIGH_NIBBLE;
                alpha[COLOR_EMP1] = GRAB_LOW_NIBBLE;
                location++;
                alpha[COLOR_PATTERN] = GRAB_HIGH_NIBBLE;
                alpha[COLOR_BACK] = GRAB_LOW_NIBBLE;
                location++;
                printf
                    ("Set Alpha (pat: 0x%X back: 0x%X emp1: 0x%X emp2: 0x%X)\n",
                     alpha[COLOR_PATTERN], alpha[COLOR_BACK],
                     alpha[COLOR_EMP1], alpha[COLOR_EMP2]);

                // force visible bitmap
                if (alpha[COLOR_PATTERN] == 0x0)
                {
                    alpha[COLOR_PATTERN] = 0xF;
                    /*
                       alpha[COLOR_BACK]=0xF;
                     */
                    alpha[COLOR_EMP1] = 0xF;
                    alpha[COLOR_EMP2] = 0xF;
                }
                break;

            case SPU_DEFINE_AREA:
                sx = GRAB_BYTE;
                location++;
                sx <<= 4;
                sx |= GRAB_HIGH_NIBBLE;
                ex = GRAB_LOW_NIBBLE;
                location++;
                ex <<= 8;
                ex |= GRAB_BYTE;
                location++;

                sy = GRAB_BYTE;
                location++;
                sy <<= 4;
                sy |= GRAB_HIGH_NIBBLE;
                ey = GRAB_LOW_NIBBLE;
                location++;
                ey <<= 8;
                ey |= GRAB_BYTE;
                location++;

                printf("Define Area (%d,%d)-(%d,%d)\n", sx, sy, ex, ey);
                width = ex + 1;
                height = ey + 1;

                break;

            case SPU_DEFINE_PIXEL_ADDRESS:
                odd_pixel_rows = GRAB_WORD;
                location += 2;
                even_pixel_rows = GRAB_WORD;
                location += 2;
                printf("Define Pixels (odd: 0x%X even: 0x%X)\n",
                       odd_pixel_rows, even_pixel_rows);

                break;

            case SPU_CHANGE_COLOR_CONTRAST:
                size = GRAB_WORD;
                location += size;
                printf("Change Color & Contrast (%d)\n", size);
                break;

            case SPU_CMD_END:
                printf("End of Commands\n");

                // handle drawing now that we got everything
                if (spu_pixelbuf)
                    free(spu_pixelbuf);
                if (!
                    (spu_pixelbuf =
                     (int *) malloc(sizeof(*spu_pixelbuf) * width * height)))
                {
                    perror("malloc");
                    exit(1);
                }
                for (i = 0; i < width * height; i++)
                    spu_pixelbuf[i] = COLOR_NOTHING;

                draw_spu(spu_pixelbuf, sy, 2, sx, buf, odd_pixel_rows);
                draw_spu(spu_pixelbuf, sy + 1, 2, sx, buf, even_pixel_rows);

                if (my_location == next_sequence)
                    return -1;



                return next_sequence;

            default:
                printf("Can't handle SPU command 0x%02X\n", cmd);
                return -1;
        }
    }
}

int spu_check(guchar * buf, int location, int max_addr)
{
    int length;

    length = GRAB_WORD;
    printf("Packet length: %d\n", length);

    if (location + length > max_addr)
        return -1;

    return location + length;
}


int main(int argc, char *argv[])
{
    GtkWidget *window, *darea;
    gint x, y;
    guchar *pos;
    guchar *spubuf;

    char *filename;
    FILE *fp;
    struct stat info;
    int spu_packet_loc;
    int spu_seq_loc;
    int packet, packet_wanted;

    gtk_init(&argc, &argv);
    gdk_rgb_init();

    // crazy purple thingy
    for (x = 0; x < 0xF; x++)
    {
        clut[x].red = 0x80;
        clut[x].green = 0x0;
        clut[x].blue = 0x80;
    }

    if (argc < 3)
    {
        printf("Usage: gtkspu subtitles.spu WHICH\n");
        exit(1);
    }

    filename = argv[1];
    packet_wanted = atoi(argv[2]);

    // suck in the whole SPU file
    if (!(fp = fopen(filename, "r")))
    {
        perror(filename);
        exit(1);
    }
    if (fstat(fileno(fp), &info) < 0)
    {
        perror("fstat");
        exit(1);
    }
    if (!(spubuf = (guchar *) malloc(info.st_size)))
    {
        perror("malloc");
    }
    if (fread(spubuf, info.st_size, 1, fp) < 1)
    {
        perror("fread");
    }
    fclose(fp);

    spu_packet_loc = 0;
    for (packet = 0; packet <= packet_wanted; packet++)
    {
        spu_seq_loc =
            spubuf[spu_packet_loc + 2] << 8 | spubuf[spu_packet_loc + 3];
        if (packet == packet_wanted)
        {
            while (spu_seq_loc != -1 && spu_packet_loc < info.st_size)
            {
                spu_seq_loc = spu_run(spubuf + spu_packet_loc, spu_seq_loc);
            }
            break;
        }
        if ((spu_packet_loc =
             spu_check(spubuf, spu_packet_loc, info.st_size)) < 0)
        {
            printf("Unfinished SPU packet %d\n", packet);
        }
    }

    if (spu_packet_loc < 0 || spu_packet_loc > info.st_size)
    {
        printf("No more packets in SPU\n");
        exit(1);
    }

    if (width == 0 || height == 0)
    {
        printf("SPU didn't have enough info about window size\n");
        exit(1);
    }

    if (!(backbuf = (guchar *) malloc(sizeof(*backbuf) * 3 * width * height)))
    {
        perror("malloc");
        exit(1);
    }
    memset(backbuf, 0xFF, 3 * width * height);

    if (!(rgbbuf = (guchar *) malloc(sizeof(*rgbbuf) * 3 * width * height)))
    {
        perror("malloc");
        exit(1);
    }
    memset(rgbbuf, 0xFF, 3 * width * height);

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    darea = gtk_drawing_area_new();
    gtk_drawing_area_size(GTK_DRAWING_AREA(darea), width, height);
    gtk_container_add(GTK_CONTAINER(window), darea);
    gtk_signal_connect(GTK_OBJECT(darea), "expose-event",
                       GTK_SIGNAL_FUNC(on_darea_expose), NULL);
    gtk_widget_show_all(window);

    /* Set up the background RGB buffer. */
    pos = backbuf;
    for (y = 0; y < height; y++)
    {
        int on = 0;
        for (x = 0; x < width; x++)
        {
            // checker board a la Gimp
            on = ((x >> 4) & 1);
            if ((y >> 4 & 1))
                on = !on;
            *pos++ = 0xFF - (on ? 80 : 0);      /* Red. */
            *pos++ = 0xFF - (on ? 80 : 0);      /* Green. */
            *pos++ = 0xFF - (on ? 80 : 0);      /* Blue. */
        }
    }

    gtk_main();
    return 0;
}


gboolean
on_darea_expose(GtkWidget * widget,
                GdkEventExpose * event, gpointer user_data)
{
    int *spu;
    guchar *screen;
    guchar *back;
    gint x, y;


    // FIXME: merge back with SPU onto rgbbuf
    spu = spu_pixelbuf;
    back = backbuf;
    screen = rgbbuf;
    for (y = 0; y < height; y++)
    {
        for (x = 0; x < width; x++)
        {
            if (*spu > COLOR_NOTHING)
            {
                printf("spu buffer freaked out (%d @ %d,%d)\n", *spu, x, y);
                exit(1);
            }
            // Red
            *screen = alpha[*spu] ? cmap[*spu].red : *back;
            screen++;
            back++;
            // Green
            *screen = alpha[*spu] ? cmap[*spu].green : *back;
            screen++;
            back++;
            // Blue
            *screen = alpha[*spu] ? cmap[*spu].blue : *back;
            screen++;
            back++;
            spu++;
        }
    }

    gdk_draw_rgb_image(widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
                       0, 0, width, height,
                       GDK_RGB_DITHER_MAX, rgbbuf, width * 3);
    return 1;
}

/* vi:set ai ts=4 sw=4 expandtab: */
